<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User;

use Exception;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Exceptions\BreakException;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Data\Models\User\UserFinder;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\User\UserBlockTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;

final class UserContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function loadSingle(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(UserTable::ID, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            if ( ! self::isAvailable($userIdFromReq))
            {
                $usernameFromReq = self::$request -> findKey(
                    UserTable::USERNAME,
                    RequestTable::PAYLOAD,
                    UserTable::TABLE_NAME
                );

                self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $usernameFromReq);

                $useUsernameValue = true;
            }
            else
            {
                $useUsernameValue = false;
            }

            /*
            |--------------------------------------------------------------------------
            |  ensure target model exists
            |--------------------------------------------------------------------------
            */

            if ( ! $useUsernameValue)
            {
                $targetUserModel = UserModel::findFromId_noException($userIdFromReq);

                self::ensureModel($targetUserModel, ResponseConstants::ENTITY_NOT_FOUND_MSG);
            }
            else
            {
                $finder = (new UserFinder())
                    -> setUsername($usernameFromReq)
                    -> find();

                if ($finder -> notFound())
                {
                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }

                $targetUserModel = $finder -> popModelFromResults();
            }

            /*
            |--------------------------------------------------------------------------
            |  is target user available
            |--------------------------------------------------------------------------
            */

            if (self::$userSession -> isAuthenticated())
            {
                if ( ! UserModelHelper::isUserAvailable($targetUserModel, self::$authedUserModel))
                {
                    throw new Exception(ResponseConstants::PRIVATE_RESULTS_MSG);
                }
            }

            /*
            |--------------------------------------------------------------------------
            |  add model to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), $targetUserModel -> getDataMap());

            /*
            |--------------------------------------------------------------------------
            |  add additional content iff current user is authenticated
            |--------------------------------------------------------------------------
            */

            if (self::$userSession -> isAuthenticated())
            {
                /*
                |--------------------------------------------------------------------------
                | additional data | user follow map
                | -------------------------------------------------------------------------
                | help build follow button
                |--------------------------------------------------------------------------
                */

                self::fetchModelsAndAddAsAdditional(
                    UserFollowTable::getTableName(),
                    array(
                        UserFollowTable::FOLLOWED_USER_ID    => $targetUserModel -> getId(),
                        UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                    )
                );

                /*
                |--------------------------------------------------------------------------
                | additional data | user block map
                | -------------------------------------------------------------------------
                | help build unblock button
                |--------------------------------------------------------------------------
                */

                self::fetchModelsAndAddAsAdditional(
                    UserBlockTable::getTableName(),
                    array(
                        UserBlockTable::BLOCKED_USER_ID    => $targetUserModel -> getId(),
                        UserBlockTable::BLOCKED_BY_USER_ID => self::$authedUserModel -> getId(),
                    )
                );
            }
        });
    }

    public static function search(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $searchTextFromReq = self::$request -> findKey(UserTable::DISPLAY_NAME, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            $offset = self::$request -> findKeyOffset(UserTable::ID, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $searchTextFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $userTableQuery = (new Query()) -> from(UserTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select search text matches
            |--------------------------------------------------------------------------
            */

            $userTableQuery
                -> openParenthesis()
                -> whereContains(UserTable::USERNAME, $searchTextFromReq)
                -> or()
                -> whereContains(UserTable::DISPLAY_NAME, $searchTextFromReq)
                -> closeParenthesis();

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $userTableQuery -> greaterThan(UserTable::ID, $offset);
            }
            else
            {
                $userTableQuery -> lessThan(UserTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $userTableQuery
                -> orderByDesc(UserTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_USER_SEARCH));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $userBeans = $userTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($userBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(UserTable::getTableName(), $userBeans, function (UserModel $userModel)
            {
                /*
                |--------------------------------------------------------------------------
                | privacy checks
                |--------------------------------------------------------------------------
                */

                if ( ! UserModelHelper::isUserAvailable($userModel, self::$authedUserModel))
                {
                    throw new BreakException();
                }
            });
        });
    }
}
